/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, RichText } from "@wordpress/block-editor";
import './style.scss';

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @param {Object} props            Properties passed to the function.
 * @param {Object} props.attributes Available block attributes.
 * @return {WPElement} Element to render.
 */
const Save = ({ attributes }) => {
  const { barVariant, colorVariant, progressNumber, blockId } = attributes;
  const blockProps = useBlockProps.save();
  let colorName = (color) => {
    switch (color) {
      case "#00598c":
        return (
          "default"
        );
        break;
      case "#ffd100":
        return (
          "gold"
        );
        break;
      case "#8bb8e8":
        return (
          "light-blue"
        );
        break;
      case "#2774ae":
        return (
          "ucla-blue"
        );
        break;
      default:
        return;
    }
  };
  return (
    <div {...blockProps}>
      <div className="ucla-progress">
          <div className="ucla-progress__info">
            <label htmlFor={`ucla-progress-${blockId}`} className="ucla-progress__label">
              <RichText.Content
            value={attributes.label}
              />
            </label>
            <div className="ucla-progress__percent">
              <RichText.Content
            value={attributes.progressValue}
              />
            </div>
          </div>
          <progress className={`ucla-progress__bar ucla-progress__bar--${barVariant} ucla-progress__bar--${colorName(colorVariant)}`} id={`ucla-progress-${blockId}`} max="100" value={progressNumber}></progress>
          <div className="ucla-progress__assistive">
            <RichText.Content
            value={attributes.message}
              />
          </div>
        </div>
    </div>
  );
};

export default Save;
