import { __ } from "@wordpress/i18n";
/**
 * WordPress components that create the necessary UI elements for the block
 *
 * @see https://developer.wordpress.org/block-editor/packages/packages-components/
 */

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, InspectorControls, RichText } from "@wordpress/block-editor";
import { PanelBody, SelectControl, ColorPalette, RangeControl, __experimentalHeading as Heading} from "@wordpress/components";
import { useEffect } from "@wordpress/element";

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @param {Object}   props               Properties passed to the function.
 * @param {Object}   props.attributes    Available block attributes.
 * @param {Function} props.setAttributes Function that updates individual attributes.
 *
 * @return {WPElement} Element to render.
 */
const Edit = ({ attributes, setAttributes, clientId }) => {
  const { barVariant, colorVariant, progressNumber, blockId } = attributes;
  const blockProps = useBlockProps();
  const onVariantChange = (variant) => {
    setAttributes({ barVariant: variant });
  };
  const onColorChange = (color) => {
    setAttributes({ colorVariant : color})
  };
  let colorName = (color) => {
    switch (color) {
      case "#00598c":
        return (
          "default"
        );
        break;
      case "#ffd100":
        return (
          "gold"
        );
        break;
      case "#8bb8e8":
        return (
          "light-blue"
        );
        break;
      case "#2774ae":
        return (
          "ucla-blue"
        );
        break;
      default:
        return;
    }
  };
  const onNumberChange = (number) => {
    setAttributes({progressNumber : number})
  };
  useEffect(()=>{
    blockId === "" && setAttributes({ blockId: clientId });
  },[])
  return (
    <>
      <InspectorControls>
        <PanelBody title={__("Settings", "ucla-wordpress-plugin")} initialOpen>
          <SelectControl
            __next40pxDefaultSize
            __nextHasNoMarginBottom
            label={__("Select variant", "ucla-wordpress-plugin")}
            value={barVariant}
            onChange={onVariantChange}
            help={__(
              "Change the variant of the progress bar.",
              "ucla-wordpress-plugin"
            )}
            options={[
              {
                value: "default",
                label: __("Default", "ucla-wordpess-plugin"),
              },
              {
                value: "thin",
                label: __("Thin", "ucla-wordpess-plugin"),
              },
              {
                value: "thick",
                label: __("Thick", "ucla-wordpess-plugin"),
              },
              {
                value: "thicker",
                label: __("Thicker", "ucla-wordpess-plugin"),
              }
            ]}
          />
          <Heading
								as="span"
								size={11}
								upperCase={true}
								weight={500}
							>Select Color</Heading>
          <ColorPalette
          colors={[
            {
              name: "Default", color: "#00598c"
            },
            {
              name: "Gold", color: "#ffd100"
            },
            {
              name:"Light Blue", color: "#8bb8e8"
            },
            {
              name:"UCLA Blue", color: "#2774ae"
            },
          ]}
          value={colorVariant}
          onChange={onColorChange}
          disableCustomColors
          />
          <RangeControl
            __next40pxDefaultSize
            __nextHasNoMarginBottom
            help="Please select how much progress has been made."
            initialPosition={50}
            label="Percentage"
            max={100}
            min={0}
            value={progressNumber}
            onChange={onNumberChange}
          />
        </PanelBody>
      </InspectorControls>
      <div {...blockProps}>
        <div className="ucla-progress">
          <div className="ucla-progress__info">
            <label htmlFor={`ucla-progress-${blockId}`} className="ucla-progress__label">
              <RichText
                  value={attributes.label}
                  onChange={(content) => {
                    setAttributes({ label: content });
                  }}
                  placeholder="Label"
                />
            </label>
            <div className="ucla-progress__percent">
              <RichText
                  value={attributes.progressValue}
                  onChange={(content) => {
                    setAttributes({ progressValue: content });
                  }}
                  placeholder="100%"
                />
            </div>
          </div>
          <progress className={`ucla-progress__bar ucla-progress__bar--${barVariant} ucla-progress__bar--${colorName(colorVariant)}`} id={`ucla-progress-${blockId}`} max="100" value={progressNumber}></progress>
          <div className="ucla-progress__assistive">
            <RichText
                  value={attributes.message}
                  onChange={(content) => {
                    setAttributes({ message: content });
                  }}
                  placeholder="Assistive text goes here"
                />
          </div>
        </div>
      </div>
    </>
  );
};

export default Edit;
