<?php

/**
 * Plugin Name:       UCLA Design System Components
 * Description:       A plugin that seamlessly integrates the UCLA Design System components into the Gutenberg editor, enabling you to build web pages effortlessly.
 * Version:           0.1.0
 * Requires at least: 6.1
 * Requires PHP:      7.0
 * Author:            UCLA Strategic Communications
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       ucla-wordpress-plugin
 *
 * @package           ucla-wordpress-plugin
 */

if (!defined('ABSPATH')) {
  wp_die(esc_html__('Do not open this file directly.', 'ucla-wordpress-plugin'));
}

/**
 * Registers the block using the metadata loaded from the `block.json` file.
 * Behind the scenes, it registers also all assets so they can be enqueued
 * through the block editor in the corresponding context.
 *
 * @see https://developer.wordpress.org/reference/functions/register_block_type/
 */
function ucla_wordpress_plugin_init()
{
  $blocks = array('alert', 'accordion', 'accordion-item', 'tile', 'ribbon', 'card', 'card-story', 'card-people', 'card-event', 'callout', 'tabs', 'tab-item', 'carousel', 'carousel-slide', 'icons', 'progress-bar');
  foreach ($blocks as $block) {
    register_block_type(__DIR__ . '/build/blocks/' . $block);
  }
}
add_action('init', 'ucla_wordpress_plugin_init');

/**
 * Extending the Block Editor
 */
function ucla_wordpress_extend_gutenberg_blocks()
{
  $dependencies = array('wp-blocks', 'wp-dom-ready');
  if (is_object(get_current_screen())) {
    if (get_current_screen()->id == 'site-editor') {
      $dependencies[] = 'wp-edit-site';
    } elseif (get_current_screen()->id == 'widgets') {
      $dependencies[] = 'wp-edit-widgets';
    } else {
      $dependencies[] = 'wp-edit-post';
    }
  } else {
    $dependencies[] = 'wp-edit-post';
  }
  $blocks = array('button', 'table', 'list', 'gallery', 'media-text', 'separator');

  foreach ($blocks as $block) {
    wp_enqueue_script(
      'ucla-' . $block . '-script',
      plugins_url('build/blocks/' . $block . '/index.js', __FILE__),
      $dependencies
    );
  }
}
add_action('enqueue_block_editor_assets', 'ucla_wordpress_extend_gutenberg_blocks');

function ucla_wordpress_extended_blocks_styles()
{

  $blocks = array('button', 'table', 'list', 'gallery', 'media-text', 'separator');
  foreach ($blocks as $block) {
    wp_enqueue_style(
      'ucla-' . $block . '-style',
      plugins_url('build/blocks/' . $block . '/style-index.css', __FILE__)
    );
    if ($block == 'media-text') {continue;}
    wp_enqueue_style(
      'ucla-' . $block . '-editor-style', // Handle.
      plugins_url('build/blocks/' . $block . '/index.css', __FILE__) // Block editor CSS.
    );
  }
}
add_action('enqueue_block_assets', 'ucla_wordpress_extended_blocks_styles');

/**
 * Creates new category in Gutenberg to sort out all Design System component blocks.
 * Uses the `block_categories_all` filter to register new categories.
 * 
 * @see https://developer.wordpress.org/reference/hooks/block_categories_all/
 */
function register_design_system_category($categories)
{
  $category_slugs = wp_list_pluck($categories, 'slug');
  return in_array('ucla-design-system-components', $category_slugs, true) ? $categories : array_merge(
    [
      [
        'slug' => 'ucla-design-system-components',
        'title' => __('UCLA Design System Components', 'ucla-wordpress-plugin'),
        'icon' => ''
      ]
    ],
    $categories
  );
};

/**
 * Patterns
 */

function ucla_wordpress_register_block_patterns()
{
  $block_pattern_categories = array(
    'banners' => array('label' => __('Banners', 'ucla-wordpress-plugin')),
    'sections' => array('label' => __('Sections', 'ucla-wordpress-plugin')),
    'full-page' => array('label' => __('Full Page', 'ucla-wordpress-plugin')),
  );
  $block_pattern_categories = apply_filters('ucla_wordpress_block_pattern_categories', $block_pattern_categories);

  foreach ($block_pattern_categories as $name => $properties) {
    if (!WP_Block_Pattern_Categories_Registry::get_instance()->is_registered($name)) {
      register_block_pattern_category($name, $properties);
    }
  }

  $block_patterns = array(
    'ribbon-text-banner',
    'text-banner',
    'ribbon-banner',
    'quote-banner',
    'box-banner',
    'text-media',
    'three-column-story-cards',
    'query-grid-cards',
    'title-hero-banner',
    'cta-banner',
    'homepage-gateway-legacy',
    'homepage-cta-layout',
    'homepage-dept-office',
    'interior-banner-sidebar',
    'interior-informative'
  );
  $block_patterns = apply_filters('ucla_wordpress_block_patterns', $block_patterns);
  foreach ($block_patterns as $block_pattern) {
    $pattern_file = __DIR__ . '/build/patterns/' . $block_pattern . '.php';

    register_block_pattern(
      'ucla-wordpress-plugin/build/' . $block_pattern,
      require $pattern_file
    );
  }
}
add_action('init', 'ucla_wordpress_register_block_patterns', 9);

require_once __DIR__ . '/build/responsive-controls/index.php';

// require_once __DIR__ . '/build/motion/index.php';



// function ucla_wordpress_enqueue_frontend_assets()
// {
//     wp_enqueue_script(
//         'ucla-block-frontend-js',
//         plugin_dir_url(__FILE__) . 'build/ucla-plugin-frontend.js',
//         array(),
//         null,
//         true
//     );
// }
// add_action('wp_enqueue_scripts', 'ucla_wordpress_enqueue_frontend_assets');

/**
 * Start motion
 */
// function ucla_wordpress_enqueue_motion() {
//   $dependencies = array('wp-blocks', 'wp-dom-ready');
//   if (is_object(get_current_screen())) {
//     if (get_current_screen()->id == 'site-editor') {
//       $dependencies[] = 'wp-edit-site';
//     } elseif (get_current_screen()->id == 'widgets') {
//       $dependencies[] = 'wp-edit-widgets';
//     } else {
//       $dependencies[] = 'wp-edit-post';
//     }
//   } else {
//     $dependencies[] = 'wp-edit-post';
//   }
//   wp_enqueue_script('ucla-motion-script', plugin_url('motion/index.js', __FILE__, ));
// }
// add_action('enqueue_block_editor_assets', function() {
//     wp_enqueue_script_module('ucla-motion', plugins_url('index.js', __FILE__),);
// });

/**
 * Enqueue asserts if block exists on site
 */
function ucla_wordpress_enqueue_if_block( $content = "" ) {
  $theme_version = wp_get_theme()->get('Version');
  $version_string = is_string($theme_version) ? $theme_version : false;
  // if ( has_block('ucla-wordpress-plugin/animation') ) {
  //   wp_enqueue_script(
  //     'motion-script',
  //     'https://cdn.jsdelivr.net/npm/motion@11.11.13/dist/motion.js',
  //     array(),
  //     $version_string,
  //     true
  //   )
  // }
  if (has_block('core/gallery')) {
      wp_enqueue_script(
        'gallery-load-more-script',
        plugins_url('build/blocks/gallery/view.js', __FILE__),
        array(),
        $version_string,
        true
      );
  }
  return $content;
}
add_filter('the_content', 'ucla_wordpress_enqueue_if_block');

/**
 * If Wordpress version is below 5.8.0
 */
if (version_compare(get_bloginfo('version'), '5.8', '>=')) {
  add_filter('block_categories_all', 'register_design_system_category');
} else {
  add_filter('block_categories', 'register_design_system_category');
}
